const SUPABASE_URL = 'https://slmmgkulowdqnorcsriq.supabase.co';
const SUPABASE_KEY = 'sb_publishable_V_8LOmKo8QYAY_jY4_lFkg_DdWzri7o';

const CURRENT_VERSION = "1.1"; // ТЕКУЩАЯ ВЕРСИЯ РАСШИРЕНИЯ
const CONFIG_URL = "https://https://data-base-ada.pages.dev/version.json"; // ТВОЯ ССЫЛКА НА CLOUDFLARE

const processedIds = new Set();
const idCache = new Map();
let pendingUsers = new Map(); 
let fetchTimer = null;

let userKey = null;
let teamId = null;
let userRole = null;
let workerName = null;

// --- СИСТЕМА КОНТРОЛЯ ВЕРСИЙ ---
const checkVersion = async () => {
    try {
        const res = await fetch(CONFIG_URL + "?t=" + Date.now()); // Защита от кэша
        const data = await res.json();
        if (data.version !== CURRENT_VERSION) {
            showUpdateScreen(data);
            return false; // Версия устарела, блокируем работу
        }
        return true; 
    } catch(e) {
        return true; // Если нет интернета, разрешаем работать
    }
};

const showUpdateScreen = (data) => {
    const overlay = document.createElement('div');
    overlay.style.cssText = "position:fixed; top:0; left:0; width:100vw; height:100vh; background:rgba(15,23,42,0.95); z-index:2147483647; display:flex; flex-direction:column; align-items:center; justify-content:center; color:white; font-family:sans-serif; backdrop-filter:blur(10px);";
    overlay.innerHTML = `
        <div style="background:#1e293b; padding:40px; border-radius:24px; border:2px solid #ef4444; text-align:center; max-width:500px; box-shadow:0 20px 25px -5px rgba(0,0,0,0.5);">
            <h1 style="font-size:24px; color:#ef4444; margin-bottom:15px; font-weight:900;">🚨 КРИТИЧЕСКОЕ ОБНОВЛЕНИЕ CRM</h1>
            <p style="font-size:16px; margin-bottom:20px; color:#cbd5e1; line-height:1.5;">${data.message}</p>
            <div style="background:#0f172a; padding:10px; border-radius:10px; margin-bottom:25px; color:#94a3b8; font-family:monospace;">
                У вас версия: <b style="color:#ef4444">${CURRENT_VERSION}</b> <br> Новая версия: <b style="color:#22c55e">${data.version}</b>
            </div>
            <a href="${data.downloadUrl}" style="display:inline-block; background:#3b82f6; color:white; padding:16px 32px; border-radius:12px; text-decoration:none; font-weight:bold; font-size:16px; box-shadow:0 4px 6px -1px rgba(59,130,246,0.5); transition:all 0.2s;">📥 СКАЧАТЬ АРХИВ</a>
            <p style="margin-top:20px; font-size:12px; color:#64748b;">Распакуйте архив с заменой файлов и обновите расширение в браузере.</p>
        </div>
    `;
    document.body.appendChild(overlay);
};

const getDeviceId = () => {
    return new Promise((resolve) => {
        chrome.storage.local.get(['deviceId'], (res) => {
            if (res.deviceId) resolve(res.deviceId);
            else {
                const newId = 'dev_' + Math.random().toString(36).substr(2, 9) + Date.now().toString(36);
                chrome.storage.local.set({ deviceId: newId }, () => resolve(newId));
            }
        });
    });
};

const host = document.createElement('div');
host.id = `crm-wrapper-${Math.random().toString(36).substr(2, 9)}`;
host.style.cssText = "position:absolute; top:0; left:0; width:100%; height:100%; pointer-events:none; z-index:2147483647;";
document.body.appendChild(host);

const shadow = host.attachShadow({ mode: 'closed' }); 
const container = document.createElement('div');
shadow.appendChild(container);

const renderedPanels = new Map();

const extractUserData = (linkElement, imgElement) => {
    if (!linkElement || !linkElement.href) return null;
    const href = linkElement.href;
    let ext_id = null;
    let ext_name = null;

    const idMatch = href.match(/id=([a-zA-Z0-9_-]+)/);
    if (idMatch) ext_id = idMatch[1];

    const pathMatch = href.match(/\/(?:user|profile|partner|member|p)\/([a-zA-Z0-9_.-]+)/i);
    if (pathMatch) {
        const val = pathMatch[1];
        if (val.toLowerCase() !== 'profile' && val.toLowerCase() !== 'user') {
            if (!ext_id) ext_id = val;       
            else ext_name = val;             
        }
    }

    if (!ext_name && imgElement) {
        const altText = imgElement.getAttribute('alt') || imgElement.getAttribute('title');
        if (altText && altText.length > 1 && altText.length < 30) {
            ext_name = altText.trim();
        }
    }

    if (!ext_id && ext_name) {
        ext_id = ext_name; 
        ext_name = null;
    }

    if (!ext_id) return null;
    return { id: ext_id, name: ext_name };
};

const fetchBatchProfiles = async (usersArray) => {
    if (usersArray.length === 0 || !userKey) return;
    try {
        let searchValues = new Set();
        usersArray.forEach(u => {
            if (u.id) searchValues.add(u.id);
            if (u.name) searchValues.add(u.name); 
        });

        const queryStr = encodeURIComponent(Array.from(searchValues).join(','));
        const res = await fetch(`${SUPABASE_URL}/rest/v1/profiles?external_id=in.(${queryStr})&select=external_id,profile_name,note`, {
            headers: { 'apikey': SUPABASE_KEY, 'Authorization': `Bearer ${SUPABASE_KEY}`, 'x-user-key': userKey }
        });
        
        if (res.ok) {
            const data = await res.json();
            data.forEach(p => {
                idCache.set(p.external_id, p);
                if (p.profile_name) idCache.set(p.profile_name, p); 
            });
            usersArray.forEach(u => updateUIPanel(u));
        }
    } catch (e) { console.error("CRM Sync Error", e); }
};

const queueDataForCheck = (userData) => {
    if (idCache.has(userData.id) || (userData.name && idCache.has(userData.name))) return;
    
    pendingUsers.set(userData.id, userData);
    clearTimeout(fetchTimer);
    fetchTimer = setTimeout(() => {
        const usersToFetch = Array.from(pendingUsers.values());
        pendingUsers.clear();
        fetchBatchProfiles(usersToFetch);
    }, 300);
};

const dbAction = async (method, body = null, pathSuffix = "") => {
    const options = {
        method,
        headers: { 'apikey': SUPABASE_KEY, 'Authorization': `Bearer ${SUPABASE_KEY}`, 'Content-Type': 'application/json', 'x-user-key': userKey, 'Prefer': 'return=minimal' }
    };
    if (body) options.body = JSON.stringify(body);
    try {
        const res = await fetch(`${SUPABASE_URL}/rest/v1/profiles${pathSuffix}`, options);
        if (!res.ok) return false;
        return true;
    } catch (e) { return false; }
};

const createUIPanel = (userData, imgElement) => {
    const panel = document.createElement('div');
    panel.style.cssText = "position:absolute; display:flex; flex-direction:column; align-items:flex-start; gap:6px; pointer-events:auto; transition: opacity 0.2s; z-index: 2;";
    
    const highlightFrame = document.createElement('div');
    const imgStyle = window.getComputedStyle(imgElement);
    const radius = imgStyle.borderRadius !== '0px' ? imgStyle.borderRadius : '8px';
    highlightFrame.style.cssText = `position:absolute; pointer-events:none; box-sizing:border-box; border-radius:${radius}; display:none; transition: all 0.3s ease; z-index: 1;`;

    const btnGroup = document.createElement('div');
    btnGroup.style.cssText = "display:flex; gap:6px;";

    const b1 = document.createElement('button');
    const b2 = document.createElement('button');
    const btnStyle = "width:32px; height:32px; border-radius:8px; border:1px solid rgba(255,255,255,0.2); background:#1e293b; color:white; cursor:pointer; font-size:14px; box-shadow:0 4px 6px -1px rgba(0,0,0,0.3); outline:none;";
    b1.style.cssText = btnStyle; b1.innerHTML = "➕"; b1.title = "В работу";
    b2.style.cssText = btnStyle; b2.innerHTML = "📝"; b2.title = "Оставить заметку";

    const noteBadge = document.createElement('div');
    noteBadge.style.cssText = "padding:3px 8px; border-radius:6px; font-size:10px; font-family:sans-serif; font-weight:bold; color:white; display:none; max-width:140px; word-wrap:break-word; text-transform:uppercase; box-shadow:0 2px 4px rgba(0,0,0,0.5); text-align: left;";

    b1.onmouseenter = () => b1.style.transform = "scale(1.1)"; b1.onmouseleave = () => b1.style.transform = "scale(1)";
    b2.onmouseenter = () => b2.style.transform = "scale(1.1)"; b2.onmouseleave = () => b2.style.transform = "scale(1)";

    const checkDbData = () => idCache.get(userData.id) || (userData.name ? idCache.get(userData.name) : null);

    b1.onclick = async (e) => {
        e.preventDefault(); e.stopPropagation();
        const dbData = checkDbData();

        if (dbData) {
            if (userRole === 'worker') return alert("Воркеры не могут удалять профили.");
            b1.innerHTML = "⏳";
            const success = await dbAction('DELETE', null, `?external_id=eq.${dbData.external_id}`);
            if (success) { 
                idCache.set(userData.id, null); 
                if (userData.name) idCache.set(userData.name, null);
                updateUIPanel(userData); 
            } else b1.innerHTML = "✅"; 
        } else {
            b1.innerHTML = "⏳";
            const success = await dbAction('POST', { 
                external_id: userData.id, 
                profile_name: userData.name || null,
                site_domain: window.location.hostname, 
                team_id: teamId, 
                note: "", 
                added_by: workerName 
            });
            if (success) { 
                const newRecord = { external_id: userData.id, profile_name: userData.name, note: "" };
                idCache.set(userData.id, newRecord); 
                if (userData.name) idCache.set(userData.name, newRecord);
                updateUIPanel(userData); 
            } else b1.innerHTML = "➕"; 
        }
    };

    b2.onclick = async (e) => {
        e.preventDefault(); e.stopPropagation();
        const dbData = checkDbData();
        const newNote = prompt("Заметка (email, block и т.д.):", dbData?.note || "");
        if (newNote !== null) {
            b2.innerHTML = "⏳";
            let success = false;
            if (dbData) success = await dbAction('PATCH', { note: newNote }, `?external_id=eq.${dbData.external_id}`);
            else success = await dbAction('POST', { 
                external_id: userData.id, 
                profile_name: userData.name || null,
                site_domain: window.location.hostname, 
                team_id: teamId, 
                note: newNote, 
                added_by: workerName 
            });
            
            if (success) { 
                const newRecord = { external_id: dbData?.external_id || userData.id, profile_name: userData.name, note: newNote };
                idCache.set(userData.id, newRecord); 
                if (userData.name) idCache.set(userData.name, newRecord);
                updateUIPanel(userData); 
            } else b2.innerHTML = "📝";
        }
    };

    btnGroup.appendChild(b1); btnGroup.appendChild(b2);
    panel.appendChild(btnGroup); panel.appendChild(noteBadge);
    container.appendChild(highlightFrame); container.appendChild(panel);

    renderedPanels.set(userData.id, { panel, highlightFrame, img: imgElement, b1, b2, noteBadge });
    queueDataForCheck(userData); 
};

const updateUIPanel = (userData) => {
    const data = renderedPanels.get(userData.id);
    if (!data) return;
    
    let dbData = idCache.get(userData.id);
    if (!dbData && userData.name) dbData = idCache.get(userData.name);

    if (dbData && dbData !== null) {
        data.b1.innerHTML = "✅"; data.b1.style.background = "#16a34a"; data.b1.style.borderColor = "#22c55e";
        if (dbData.note) {
            let displayNote = dbData.note;
            if (userRole === 'worker') displayNote = displayNote.replace(/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/g, '📧 [СКРЫТЫЙ EMAIL]');
            
            data.noteBadge.innerText = displayNote; data.noteBadge.style.display = "block";
            const isBlock = dbData.note.toLowerCase().includes('блок') || dbData.note.toLowerCase().includes('block');
            
            data.noteBadge.style.background = isBlock ? "#dc2626" : "#2563eb"; data.b2.style.background = "#2563eb";
            data.highlightFrame.style.border = `4px solid ${isBlock ? '#dc2626' : '#2563eb'}`;
            data.highlightFrame.style.boxShadow = `inset 0 0 20px ${isBlock ? '#dc2626' : '#2563eb'}, 0 0 15px ${isBlock ? '#dc2626' : '#2563eb'}`;
        } else {
            data.noteBadge.style.display = "none"; data.b2.style.background = "#1e293b";
            data.highlightFrame.style.border = "4px solid #16a34a";
            data.highlightFrame.style.boxShadow = "inset 0 0 20px rgba(22, 163, 74, 0.8), 0 0 15px rgba(22, 163, 74, 0.8)";
        }
        data.highlightFrame.style.display = 'block'; 
    } else {
        data.b1.innerHTML = "➕"; data.b1.style.background = "#1e293b"; data.b1.style.borderColor = "rgba(255,255,255,0.2)";
        data.noteBadge.style.display = "none"; data.b2.style.background = "#1e293b"; data.highlightFrame.style.display = 'none'; 
    }
};

const updatePositions = () => {
    renderedPanels.forEach(({ panel, highlightFrame, img }) => {
        const rect = img.getBoundingClientRect();
        if (rect.width === 0 || rect.top > window.innerHeight || rect.bottom < 0) {
            panel.style.display = 'none'; highlightFrame.style.display = 'none';
        } else {
            panel.style.display = 'flex';
            panel.style.top = `${rect.top + window.scrollY + 6}px`; 
            panel.style.left = `${rect.left + window.scrollX + 6}px`;
            
            highlightFrame.style.top = `${rect.top + window.scrollY}px`; highlightFrame.style.left = `${rect.left + window.scrollX}px`;
            highlightFrame.style.width = `${rect.width}px`; highlightFrame.style.height = `${rect.height}px`;
            if (highlightFrame.style.border !== "transparent" && highlightFrame.style.border !== "") highlightFrame.style.display = 'block';
        }
    });
};

const scanPage = () => {
    const be2Domains = ['zusammen', 'be2', 'kultiviertesingles', 'verliebtab40', '60plustreff', '50plus-singleboerse', 'aktiv-und-verliebt'];
    const isBe2Network = be2Domains.some(d => window.location.hostname.includes(d));

    if (isBe2Network) {
        const profileViews = document.querySelectorAll('profile-view, .state-profile-layout1');
        profileViews.forEach(pv => {
            let nameEl = pv.querySelector('.name-label');
            let ageEl = pv.querySelector('.age-height');
            let locEl = pv.querySelector('.subline-profile-from');
            let imgEl = pv.querySelector('img');

            if (nameEl && ageEl && locEl && imgEl) {
                let clone = nameEl.cloneNode(true);
                let online = clone.querySelector('.is-online');
                if (online) online.remove();
                let name = clone.textContent.replace(/[•]/g, '').trim();

                let ageMatch = ageEl.textContent.match(/(\d{2})/);
                let age = ageMatch ? ageMatch[1] : '';

                let locParts = locEl.textContent.split(',');
                let loc = locParts[locParts.length - 1].trim();

                if (name && age && loc) {
                    let compositeId = `${name}_${age}_${loc}`.replace(/\s+/g, '_').toLowerCase();
                    let userData = { id: compositeId, name: name };
                    if (!processedIds.has(userData.id)) {
                        processedIds.add(userData.id);
                        createUIPanel(userData, imgEl);
                    }
                }
            }
        });

        const gridImgs = document.querySelectorAll('img');
        gridImgs.forEach(img => {
            if (img.width < 50 || img.height < 50) return; 

            let card = img.parentElement;
            let foundText = false;
            for(let i = 0; i < 4; i++) {
                if (card && card.textContent && card.textContent.match(/\d{2}\s*[,•]/)) {
                    foundText = true;
                    break;
                }
                if (card) card = card.parentElement;
            }

            if (foundText && card) {
                const text = card.textContent.replace(/\s+/g, ' ').trim();
                const match = text.match(/([a-zA-ZäöüÄÖÜß\-]+).*?(\d{2})\s*[,•]\s*([a-zA-ZäöüÄÖÜß\s\-]+)/);
                
                if (match && !card.dataset.crmParsed) {
                    let name = match[1].trim();
                    let age = match[2].trim();
                    let loc = match[3].trim();
                    
                    let compositeId = `${name}_${age}_${loc}`.replace(/\s+/g, '_').toLowerCase();
                    let userData = { id: compositeId, name: name };

                    card.dataset.crmParsed = 'true'; 

                    if (!processedIds.has(userData.id)) {
                        processedIds.add(userData.id);
                        createUIPanel(userData, img);
                    }
                }
            }
        });
    }

    if (!isBe2Network) {
        const selectors = ['img.object-cover', 'img[data-test="avatar"]', 'img.photo', 'img.profile-image', 'img.user-image', '.profile-photo img', '.avatar img', 'a img'];
        const imgs = document.querySelectorAll(selectors.join(', '));
        imgs.forEach(img => {
            const link = img.closest('a');
            const userData = extractUserData(link, img);
            
            if (userData && !processedIds.has(userData.id)) { 
                processedIds.add(userData.id); 
                createUIPanel(userData, img); 
            }
        });
    }
    
    updatePositions();
};

chrome.storage.local.get(['apiKey'], async (res) => {
    if (!res.apiKey) return;
    
    // ПЕРВЫМ ДЕЛОМ ПРОВЕРЯЕМ ВЕРСИЮ!
    const isUpToDate = await checkVersion();
    if (!isUpToDate) return; // Останавливаем загрузку расширения, если оно устарело

    userKey = res.apiKey;

    const authRes = await fetch(`${SUPABASE_URL}/rest/v1/api_keys?select=role,team_id,worker_name,key_value`, {
        headers: { 'apikey': SUPABASE_KEY, 'Authorization': `Bearer ${SUPABASE_KEY}`, 'x-user-key': userKey }
    });
    
    if (!authRes.ok) return console.error("CRM: Invalid Key");
    const authData = await authRes.json();
    if (!authData.length) return;
    
    teamId = authData[0].team_id;
    userRole = authData[0].role;
    workerName = authData[0].worker_name || authData[0].key_value; 

    if (userRole === 'worker') {
        const today = new Date().toISOString().split('T')[0];
        const deviceId = await getDeviceId();

        await fetch(`${SUPABASE_URL}/rest/v1/device_logs`, {
            method: 'POST',
            headers: { 'apikey': SUPABASE_KEY, 'Authorization': `Bearer ${SUPABASE_KEY}`, 'Content-Type': 'application/json', 'Prefer': 'resolution=ignore-duplicates' },
            body: JSON.stringify({ key_value: userKey, device_id: deviceId, logged_date: today })
        });

        const countRes = await fetch(`${SUPABASE_URL}/rest/v1/device_logs?select=device_id&key_value=eq.${userKey}&logged_date=eq.${today}`, {
            headers: { 'apikey': SUPABASE_KEY, 'Authorization': `Bearer ${SUPABASE_KEY}` }
        });
        
        const countData = await countRes.json();
        if (countData.length > 45) {
            alert("🚨 DATING CRM: ДОСТУП ПРИОСТАНОВЛЕН 🚨\n\nПревышен дневной лимит профилей (45 шт). Субадмину отправлено уведомление.");
            await fetch(`${SUPABASE_URL}/functions/v1/telegram-bot`, {
                method: 'POST', headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ type: 'LEAK_ALERT', key: userKey, team: teamId, worker: workerName, count: countData.length })
            });
            return; 
        }
    }

    scanPage();
    const observer = new MutationObserver((mutations) => {
        let hasNewNodes = mutations.some(m => m.addedNodes.length > 0);
        if (hasNewNodes) requestAnimationFrame(scanPage);
    });
    observer.observe(document.body, { childList: true, subtree: true });

    window.addEventListener('scroll', () => requestAnimationFrame(updatePositions), { passive: true });
    window.addEventListener('resize', () => requestAnimationFrame(updatePositions));
});